from django.core.mail import send_mail
from django.conf import settings
from rest_framework.response import Response
from rest_framework import status
from rest_framework.views import APIView
from rest_framework.permissions import AllowAny, IsAuthenticated
from rest_framework_simplejwt.tokens import RefreshToken, TokenError
import ssl

ssl._create_default_https_context = ssl._create_unverified_context  # SSL bypass for local dev

from .models import CustomUser
from .serializers import RegisterSerializer, LoginSerializer


# JWT utility
def get_tokens_for_user(user):
    refresh = RefreshToken.for_user(user)
    return {"refresh": str(refresh), "access": str(refresh.access_token)}


class RegisterView(APIView):
    permission_classes = [AllowAny]

    def post(self, request):
        serializer = RegisterSerializer(data=request.data)
        if serializer.is_valid():
            user = serializer.save()

            # Explicitly set is_admin if provided in the request
            user.is_admin = request.data.get("is_admin", False)
            user.is_customer = not user.is_admin
            user.save()

            tokens = get_tokens_for_user(user)

            # Send welcome email
            email_status = "Email not sent"
            if user.email:
                try:
                    send_mail(
                        subject="Welcome to Cranes Adventure",
                        message=f"Hello {user.first_name or user.email},\n\nThank you for registering!",
                        from_email=settings.EMAIL_HOST_USER,
                        recipient_list=[user.email],
                        fail_silently=False,
                    )
                    email_status = "Email sent successfully"
                except Exception as e:
                    email_status = f"Email sending failed: {str(e)}"

            return Response(
                {
                    "message": "User registered successfully",
                    "user": {
                        "id": user.id,
                        "first_name": user.first_name,
                        "last_name": user.last_name,
                        "email": user.email,
                        "phone": getattr(user, "phone", None),
                        "address": getattr(user, "address", None),
                        "is_admin": user.is_admin,
                        "is_customer": user.is_customer,
                    },
                    "tokens": tokens,
                    "email_status": email_status,
                },
                status=status.HTTP_201_CREATED,
            )

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class LoginView(APIView):
    permission_classes = [AllowAny]

    def post(self, request):
        serializer = LoginSerializer(data=request.data)
        if serializer.is_valid():
            user = serializer.validated_data["user"]
            tokens = get_tokens_for_user(user)
            return Response(
                {
                    "message": "Login successful",
                    "user": {
                        "id": user.id,
                        "first_name": user.first_name,
                        "last_name": user.last_name,
                        "email": user.email,
                        "phone": getattr(user, "phone", None),
                        "address": getattr(user, "address", None),
                        "is_admin": user.is_admin,
                        "is_customer": user.is_customer,
                    },
                    "tokens": tokens,
                },
                status=status.HTTP_200_OK,
            )

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class CurrentUserView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        user = request.user
        return Response(
            {
                "id": user.id,
                "first_name": user.first_name,
                "last_name": user.last_name,
                "email": user.email,
                "phone": getattr(user, "phone", None),
                "address": getattr(user, "address", None),
                "is_admin": user.is_admin,
                "is_customer": user.is_customer,
            },
            status=status.HTTP_200_OK,
        )


class UpdateUserView(APIView):
    permission_classes = [IsAuthenticated]

    def put(self, request):
        user = request.user
        data = request.data

        user.first_name = data.get("first_name", user.first_name)
        user.last_name = data.get("last_name", user.last_name)
        user.email = data.get("email", user.email)
        user.phone = data.get("phone", user.phone)
        user.address = data.get("address", user.address)

        if "password" in data and data["password"]:
            user.set_password(data["password"])

        user.save()

        return Response(
            {
                "message": "User updated successfully",
                "user": {
                    "id": user.id,
                    "first_name": user.first_name,
                    "last_name": user.last_name,
                    "email": user.email,
                    "phone": getattr(user, "phone", None),
                    "address": getattr(user, "address", None),
                    "is_admin": user.is_admin,
                    "is_customer": user.is_customer,
                },
            },
            status=status.HTTP_200_OK,
        )


class DeleteUserView(APIView):
    permission_classes = [IsAuthenticated]

    def delete(self, request):
        user = request.user
        user.delete()
        return Response({"message": "User deleted successfully"}, status=status.HTTP_200_OK)


class AdminDeleteUserView(APIView):
    permission_classes = [IsAuthenticated]

    def delete(self, request, user_id):
        if not request.user.is_admin:
            return Response({"error": "Only admins can delete users."}, status=status.HTTP_403_FORBIDDEN)

        try:
            user = CustomUser.objects.get(id=user_id)
            user.delete()
            return Response({"message": "User deleted successfully"}, status=status.HTTP_200_OK)
        except CustomUser.DoesNotExist:
            return Response({"error": "User not found."}, status=status.HTTP_404_NOT_FOUND)


class FetchAllUsersView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        if not request.user.is_admin:
            return Response({"error": "Only admins can view all users."}, status=status.HTTP_403_FORBIDDEN)

        users = CustomUser.objects.all()
        users_data = [
            {
                "id": user.id,
                "first_name": user.first_name,
                "last_name": user.last_name,
                "email": user.email,
                "phone": getattr(user, "phone", None),
                "address": getattr(user, "address", None),
                "is_admin": user.is_admin,
                "is_customer": user.is_customer,
            }
            for user in users
        ]

        return Response({"users": users_data}, status=status.HTTP_200_OK)


class LogoutView(APIView):
    permission_classes = [IsAuthenticated]

    def post(self, request):
        refresh_token = request.data.get("refresh")
        if not refresh_token:
            return Response(
                {"error": "Refresh token is required"},
                status=status.HTTP_400_BAD_REQUEST,
            )

        try:
            token = RefreshToken(refresh_token)

            try:
                token.blacklist()
            except AttributeError:
                pass

            return Response(
                {"message": "Logout successful"},
                status=status.HTTP_205_RESET_CONTENT,
            )
        except TokenError:
            return Response(
                {"error": "Invalid or expired refresh token"},
                status=status.HTTP_400_BAD_REQUEST,
            )
