from django.db import models

class Package(models.Model):
    package = models.CharField(max_length=255)
    adults = models.PositiveIntegerField(null=True, blank=True)
    children = models.PositiveIntegerField(null=True, blank=True)
    nights = models.PositiveIntegerField(null=True, blank=True)
    days = models.PositiveIntegerField(null=True, blank=True)

    total = models.DecimalField(max_digits=12, decimal_places=2, null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return self.package

    def calculate_total(self):
        """
        Total = sum of all itinerary day costs
        (adults × price_per_adult) + (children × price_per_child)
        """
        total = 0
        for day in self.itinerary_days.all():
            if self.adults and day.price_per_adult:
                total += self.adults * float(day.price_per_adult)
            if self.children and day.price_per_child:
                total += self.children * float(day.price_per_child)
        return total

    def save(self, *args, **kwargs):
        # Auto-update total every time the package is saved
        super().save(*args, **kwargs)  # save first (so we have an ID for related days)
        self.total = self.calculate_total()
        super().save(update_fields=["total"])


class ItineraryDay(models.Model):
    package = models.ForeignKey(
        Package, on_delete=models.CASCADE, related_name="itinerary_days"
    )
    day = models.PositiveIntegerField()
    title = models.CharField(max_length=255)
    description = models.TextField(blank=True, null=True)

    # Prices stored per day
    price_per_adult = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)
    price_per_child = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)

    class Meta:
        unique_together = ("package", "day")
        ordering = ["day"]

    def __str__(self):
        return f"{self.package.package} - Day {self.day}: {self.title}"

    def save(self, *args, **kwargs):
        super().save(*args, **kwargs)
        # update parent package total after saving a day
        self.package.total = self.package.calculate_total()
        self.package.save(update_fields=["total"])
